//Arduino Leonardo or Pro Micro using ATMega32u4    (SerialControlAnalogDev9)
//Serial input control; serial data output streaming; analog light sensor
//Pulse duration measurement; Pulse onset latency measurement
//Place 100 nF capacitor between A0 and GND to reduce noise

const int signalPin = A0;       //phototransistor light detector
const int externalLED = 9;      //external LED output (RED)
const int redLED = 5;           //RED pin of RGB LED (Active = LOW)
const int greenLED = 4;         //GREEN pin of RGB LED (Active = LOW)
const int OLFACTORY = 8;        //Olfacory spray output
const int digitalInput = 7;     //digital input #1
const int digitalInput2 = 6;    //digital input #2
unsigned long startTime = 0UL;
unsigned long endTime = 0UL;
unsigned long elapsedTime = 0UL;
unsigned long TIMEOUT = 2000;   //RT trial timeout (2000 millisecs)

char data;                   //serial char input data
int loopState;               //loop control flag
int button;                  //2AFC RT button pressed (0,1 or 2)

//operalional parameters
int state = 0;            //operational state: 0=stopped,1=duration,2=latency
int trigger = 0;          //event detection: 0=rising-edge, 1=falling-edge (N/A)
int verbose = 0;          //0=silent, 1=serial output commentary enabled+
int black = 20;           //BLACK photometer level (default)
int white = 100;          //WHITE photometer level (default)
int threshold = 60;       //photometer event detection threshold
int light = 0;            //analog light sensor level
unsigned int microsecs = 500; //ADC delay (microsecs)

void setup() {
  pinMode(externalLED, OUTPUT); //define external LED output
  pinMode(redLED, OUTPUT);      //define RED (RGB) LED
  pinMode(greenLED, OUTPUT);    //define GREEN (RGB) LED output
  pinMode(OLFACTORY, OUTPUT);   //define OLFACTORY SPRAY output
  pinMode(signalPin, INPUT);    //define photometeer analog input
  pinMode(digitalInput, INPUT_PULLUP);  //define digital input #1
  pinMode(digitalInput2, INPUT_PULLUP); //define digital input #2
  
  digitalWrite(externalLED, LOW); //RED LED off
  digitalWrite(redLED, HIGH);     //RED (RGB) LED off
  digitalWrite(greenLED, HIGH);   //GREEN (RGB) LED off
  digitalWrite(OLFACTORY, LOW);   //GREEN LED/OLFACTORY spray off
  
  analogReference(INTERNAL);    //use ATMega324u's internal 2.56V reference for ADC
  for(int i=0; i<10; i++) {     //accommodate change in ADC reference connection
     int temp = analogRead(signalPin);
     delay(50);
  }
  elapsedTime = micros();
  Serial.begin(115200);
}

void loop() {

  int loopflag;
  int temp;
  
  ////////////////////////////////////////
  //                                    //
  // posiive-pulse duration measurement //
  //                                    //
  ////////////////////////////////////////
  if( (state==1) && (trigger == 0)) {
    //wait for initial low input signal level
    if(verbose==1) {Serial.println("Waiting for baseline low signal level");}
    loopflag = 1;
    while(loopflag) {
      delayMicroseconds(microsecs);
      temp = analogRead(signalPin);
      if(temp < threshold) {
         loopflag = 0;
         //Serial.print("LOW LEVEL FOUND   "); Serial.println(temp);
      }
      else
      {
         //Serial.print("searching for LOW signal   "); Serial.println(temp);
         scanKey();
      }
      if(state != 1) { break; }
    }
    
    //wait for initial rising-edge
    if(verbose==1) {Serial.println("Scan for initial rising-edge of signal");}
    loopflag = 1;
    while(loopflag) {
      delayMicroseconds(microsecs);
      temp = analogRead(signalPin);
      if(temp > threshold) {
         startTime = micros();  /////// READ EVENT START TIMEE
         //Serial.print("INITIAL POS EDGE DETECTED   "); Serial.println(temp);
         //Serial.println("Scaning for negative-edge   ");
         loopflag = 0;
      }
      else
      {
         //Serial.print("searching for positive edge   "); Serial.println(temp);
         scanKey();
      }
      if(state != 1) { break; }
    }
    
    //wait for negative-edge
    loopflag = 1;
    while(loopflag) {
       delayMicroseconds(microsecs);
       temp = analogRead(signalPin);
       if(temp < threshold) {
          endTime = micros();
          //Serial.print("END OF EVENT DETECTED   "); Serial.println(temp);
          loopflag = 0;
       }
       else
       {
          //Serial.print("searching for negative-edge   "); Serial.println(temp);
          scanKey();
       }
       if(state != 1) { break; }   //end one-shot mode
    }

    //compute and return positive pulse duration
    elapsedTime = endTime - startTime;
    if(state == 1) {Serial.println(elapsedTime);}

    //terminate pulse duration measurement
    //digitalWrite(ledPin, LOW);  //LED off
    state = 0;
    
  } //end one-shot, rising-edge


  ////////////////////////////////////////
  //                                    //
  // posiive-pulse latency measurement  //
  //                                    //
  ////////////////////////////////////////
  if( (state==2) && (trigger == 0)) {

    startTime = micros();   //save scan start time
    //wait for initial rising-edge
    if(verbose==1) {Serial.println("Scan for initial rising-edge of signal");}
    loopflag = 1;
    while(loopflag) {
      delayMicroseconds(microsecs);
      temp = analogRead(signalPin);
      if(temp > threshold) {
         endTime = micros();  //read event detection time
         //Serial.print("POS EDGE DETECTED   "); Serial.println(temp);
         loopflag = 0;
      }
      else
      {
         //Serial.print("searching for positive edge   "); Serial.println(temp);
         scanKey();
      }
      if(state != 2) { break; }
    }

    //compute and return pulse latency
    elapsedTime = endTime - startTime;
    if(state == 2) {Serial.println(elapsedTime);}

    //terminate pulse latency measurement
    //digitalWrite(ledPin, LOW);  //LED off
    state = 0;
    
  } //end of latency scan loop


  //scan for serial input commands
  scanKey();
  
} //end loop()

/////////////////////////////////////////////////////////
//                                                     //
//scan serial input and interpret single-char commands //
//                                                     //
/////////////////////////////////////////////////////////
void scanKey() {
   int min = 1023;
   int max = 0;
   if(Serial.available() > 0) {
       data = Serial.read();   //fetch char from serial input buffer
       //interpret serial input char
       switch(data) {
          case 'p':   //(P)ing command (used for baseline timing benchmarks)
             Serial.print("R");  //immediate single char serial reply
             break;
          case 's':   //(S)top command
             //digitalWrite(ledPin, LOW);
             if(verbose==1) {Serial.println("Measurement stopped"); }
             state = 0;   //operational mode = stopped
             break;
          case 'd':   //(D)uration measurement command
             //digitalWrite(ledPin, HIGH);
             state = 1;   //operational mode = one-shot measurement
             if(verbose==1) {Serial.println("One-shot mode"); }
             break;
          case 'l':   //(L)atency measurement command
             //digitalWrite(ledPin, HIGH);
             if(verbose==1) {Serial.println("Contimuous mode"); }
             state = 2;   //operational mode = continuous measurement
             break;
          case 'b':   //sample stimulus BLACK level and update threshold value
             //int min = 1023;
             //int max = 0;
             Serial.println("Sampling...");
             for(int i=0; i<20; i++) {
               delay(103);
               int temp = analogRead(signalPin);
               //Serial.print("ADC = "); Serial.println(temp);
               if(temp < min) min = temp;
               if(temp > max) max = temp;
               //delay(103);
             }
             black = max;
             threshold = white - ((white-black)/2);
             Serial.print("Black = "); Serial.print(black);
             Serial.print(" Threshold = "); Serial.println(threshold);
             break;
          case 'w':   //sample stimulus WHITE level and update threshold value
             //int min = 1023;
             //int max = 0;
             Serial.println("Sampling...");
             for(int i=0; i<20; i++) {
                delay(103);
                int temp = analogRead(signalPin);
                //Serial.print("ADC = "); Serial.println(temp);
                if(temp < min) min = temp;
                if(temp > max) max = temp;
                //delay(103);
             }
             white = min;
             threshold = white - ((white-black)/2);
             Serial.print("White = "); Serial.print(white);
             Serial.print(" Threshold = "); Serial.println(threshold);
             break;   
          case 'a':   //ADC reading of light sensor input command
             light = analogRead(signalPin);
             Serial.print("Light level = "); Serial.println(light);
             break;
          case 't':   //(T)HRESHOLD report command
             Serial.print("Threshold = "); Serial.print(threshold);
             Serial.print(" Black = "); Serial.print(black);
             Serial.print(" White = "); Serial.println(white);
             break;
          case 'c':   //(C)lock
             startTime = micros();
             Serial.println(startTime);
             break;
          case '0':   //Turn external LED off
             digitalWrite(externalLED, LOW);
             break;
          case '1':   //Turn external LED On
             digitalWrite(externalLED, HIGH);
             break;
          case '6':   //Report state of Digital Input #2
             //Serial.print("D6 = "); 
             Serial.println(digitalRead(digitalInput2));
             break;
          case '7':   //Report state of Digital Input #1
             //Serial.print("D7 = "); 
             Serial.println(digitalRead(digitalInput));
             break;
          case 'o':   //Olfactory spray command
             digitalWrite(OLFACTORY, HIGH);
             delay(200);
             digitalWrite(OLFACTORY, LOW);
             break;
          case 'r':   //pulse (R)ED RGB feedback LED
             digitalWrite(redLED, LOW);
             delay(1000);
             digitalWrite(redLED, HIGH);
             break;
          case 'g':   //pulse (G)REEN RGB feedback LED
             digitalWrite(greenLED, LOW);
             delay(1000);
             digitalWrite(greenLED, HIGH);
             break; 
          case 'i':   //scan (I)nputs for 2AFC RT response
             startTime = millis();
             button = 0;
             loopState = 1;
             //loop until response or timeout
             while(loopState == 1) {
                //scan button #1
                if(digitalRead(digitalInput) == LOW) {
                   endTime = millis();
                   button = 1;
                   loopState = 0;
                }
                //scan button #2
                if(digitalRead(digitalInput2) == LOW) {
                   endTime = millis();
                   button = 2;
                   loopState = 0;
                }
                //test for timeout condition
                endTime = millis();
                if( (endTime-startTime) >= TIMEOUT) {
                   button = 0;
                   loopState = 0;
                }
             }
             elapsedTime = endTime-startTime;
             Serial.println(elapsedTime);
             Serial.println(button);
             break;
          case 'h':   //(H)ELP menu command
          case '?':
             Serial.println("Single Character Commands (version 9):");
             Serial.println("  a = ADC light level (analog photo-sensor)");
             Serial.println("  b = sample stimulus BLACK level");
             Serial.println("  c = Return Arduino Clock (microsecs)");
             Serial.println("  d = Duration measurement mode");
             Serial.println("  g = pulse GREEN feedback LED");
             Serial.println("  i = scan 2AFC inputs for RT response");
             Serial.println("  l = Latency measurement mode");
             Serial.println("  o = Olfactory spray trigger");
             Serial.println("  p = Ping the Arduino (immediate serial reply");
             Serial.println("  r = pulse RED feedback LED");
             Serial.println("  s = Stop ongoing measurement operations");
             Serial.println("  t = report Threshold level");
             Serial.println("  w = sample stimulus WHITE level");
             Serial.println("  0 = external LED Off");
             Serial.println("  1 = external LED On");
             Serial.println("  6 = read state of Digital Input #2 (D6)");
             Serial.println("  7 = read state of Digital Input #1 (D7)");
             break;
          default:
             break;
       } //end switch()
    } //end Serial.available()
} //end scanKey()
